<?php
require_once 'config/db.php';

$conn = getDBConnection();
$success = '';
$error = '';

// Handle review submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $name = sanitize($conn, $_POST['name'] ?? '');
    $email = sanitize($conn, $_POST['email'] ?? '');
    $rating = intval($_POST['rating'] ?? 0);
    $review_text = sanitize($conn, $_POST['review'] ?? '');
    
    // Handle image upload
    $image_path = '';
    if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['image']['name'];
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        if (in_array($ext, $allowed)) {
            $new_filename = uniqid() . '.' . $ext;
            $upload_path = 'assets/uploads/reviews/';
            
            if (!is_dir($upload_path)) {
                mkdir($upload_path, 0777, true);
            }
            
            if (move_uploaded_file($_FILES['image']['tmp_name'], $upload_path . $new_filename)) {
                $image_path = 'reviews/' . $new_filename;
            }
        }
    }
    
    if (empty($name) || empty($review_text) || $rating < 1 || $rating > 5) {
        $error = "Please fill in all required fields and provide a rating.";
    } else {
        $stmt = $conn->prepare("INSERT INTO reviews (customer_name, customer_email, rating, review_text, image, status) VALUES (?, ?, ?, ?, ?, 'pending')");
        $stmt->bind_param("ssiss", $name, $email, $rating, $review_text, $image_path);
        
        if ($stmt->execute()) {
            $success = "Thank you for your review! It will be visible after admin approval.";
        } else {
            $error = "Something went wrong. Please try again.";
        }
    }
}

// Get approved reviews
$reviews = [];
$result = $conn->query("SELECT * FROM reviews WHERE status = 'approved' ORDER BY created_at DESC");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $reviews[] = $row;
    }
}

// Get site settings
$settings = [];
$result = $conn->query("SELECT * FROM site_settings");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customer Reviews - FRUITOLA | Daily Fresh Dose</title>
    <meta name="description" content="Read what our customers say about FRUITOLA. Share your experience with us.">
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .star-rating-input {
            display: flex;
            gap: 10px;
            font-size: 2.5rem;
            color: #ddd;
            cursor: pointer;
        }
        .star-rating-input .star {
            transition: all 0.2s ease;
        }
        .star-rating-input .star:hover,
        .star-rating-input .star.active {
            color: #f39c12;
            transform: scale(1.1);
        }
        .review-image {
            max-width: 200px;
            border-radius: 10px;
            margin-top: 15px;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="navbar-container">
            <a href="index.php" class="logo">
                <span class="logo-text">FRUITOLA</span>
            </a>
            <ul class="nav-links" id="navLinks">
                <li><a href="index.php">Home</a></li>
                <li><a href="menu.php">Menu</a></li>
                <li><a href="reviews.php" class="active">Reviews</a></li>
                <li><a href="contact.php">Contact</a></li>
                <li><a href="roadmap.php">Roadmap</a></li>
            </ul>
            <div class="mobile-menu-btn" onclick="toggleMenu()">
                <span></span>
                <span></span>
                <span></span>
            </div>
        </div>
    </nav>

    <!-- Page Header -->
    <section class="page-header">
        <div class="page-header-content">
            <h1 class="page-title">Customer Reviews</h1>
            <p class="page-subtitle">See what our customers are saying about us</p>
        </div>
    </section>

    <!-- Review Form Section -->
    <section class="section" style="background: linear-gradient(135deg, #fff9f5 0%, #f0fff4 100%);">
        <div class="section-header">
            <h2 class="section-title">
                Share Your <span class="highlight">Experience</span>
            </h2>
            <p class="section-subtitle">
                We value your feedback! Write a review and let us know how we're doing
            </p>
        </div>
        
        <div class="review-form-container">
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo $success; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="" enctype="multipart/form-data">
                <div class="form-group">
                    <label>Your Name *</label>
                    <input type="text" name="name" required placeholder="Enter your name">
                </div>
                
                <div class="form-group">
                    <label>Your Email</label>
                    <input type="email" name="email" placeholder="Enter your email (optional)">
                </div>
                
                <div class="form-group">
                    <label>Your Rating *</label>
                    <div class="star-rating-input" id="starRating">
                        <span class="star" data-rating="1"><i class="fas fa-star"></i></span>
                        <span class="star" data-rating="2"><i class="fas fa-star"></i></span>
                        <span class="star" data-rating="3"><i class="fas fa-star"></i></span>
                        <span class="star" data-rating="4"><i class="fas fa-star"></i></span>
                        <span class="star" data-rating="5"><i class="fas fa-star"></i></span>
                    </div>
                    <input type="hidden" name="rating" id="ratingValue" required>
                </div>
                
                <div class="form-group">
                    <label>Your Review *</label>
                    <textarea name="review" rows="5" required placeholder="Share your experience with us..."></textarea>
                </div>
                
                <div class="form-group">
                    <label>Upload Image (Optional)</label>
                    <input type="file" name="image" accept="image/*">
                    <small style="color: #666; display: block; margin-top: 5px;">Max file size: 2MB. Allowed formats: JPG, PNG, GIF</small>
                </div>
                
                <button type="submit" class="btn btn-primary" style="width: 100%;">
                    <i class="fas fa-paper-plane"></i> Submit Review
                </button>
            </form>
        </div>
    </section>

    <!-- Reviews Display Section -->
    <section class="section reviews-section">
        <div class="section-header">
            <h2 class="section-title">
                What Our <span class="highlight">Customers</span> Say
            </h2>
            <p class="section-subtitle">
                Real reviews from real fruit lovers
            </p>
        </div>
        
        <div class="reviews-grid">
            <?php if (!empty($reviews)): ?>
                <?php foreach ($reviews as $review): ?>
                <div class="review-card">
                    <div class="review-header">
                        <img src="https://ui-avatars.com/api/?name=<?php echo urlencode($review['customer_name']); ?>&background=f39c12&color=fff" alt="<?php echo htmlspecialchars($review['customer_name']); ?>" class="review-avatar">
                        <div class="review-info">
                            <h4><?php echo htmlspecialchars($review['customer_name']); ?></h4>
                            <div class="review-stars">
                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                    <i class="fas fa-star<?php echo $i <= $review['rating'] ? '' : '-o'; ?>"></i>
                                <?php endfor; ?>
                            </div>
                        </div>
                    </div>
                    <p class="review-text">"<?php echo htmlspecialchars($review['review_text']); ?>"</p>
                    <?php if ($review['image']): ?>
                    <img src="assets/uploads/<?php echo $review['image']; ?>" alt="Review Image" class="review-image">
                    <?php endif; ?>
                    <small style="color: #999; display: block; margin-top: 15px;">
                        <i class="fas fa-calendar"></i> <?php echo date('M d, Y', strtotime($review['created_at'])); ?>
                    </small>
                </div>
                <?php endforeach; ?>
            <?php else: ?>
                <!-- Default reviews -->
                <div class="review-card">
                    <div class="review-header">
                        <img src="https://ui-avatars.com/api/?name=Priya+S&background=f39c12&color=fff" alt="Priya S" class="review-avatar">
                        <div class="review-info">
                            <h4>Priya Sharma</h4>
                            <div class="review-stars">
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                            </div>
                        </div>
                    </div>
                    <p class="review-text">"Absolutely love FRUITOLA! Their fruit bowls are so fresh and delicious. The Tropical Paradise is my favorite. Highly recommend!"</p>
                    <small style="color: #999; display: block; margin-top: 15px;">
                        <i class="fas fa-calendar"></i> Jan 15, 2026
                    </small>
                </div>
                
                <div class="review-card">
                    <div class="review-header">
                        <img src="https://ui-avatars.com/api/?name=Rahul+M&background=e74c3c&color=fff" alt="Rahul M" class="review-avatar">
                        <div class="review-info">
                            <h4>Rahul Mehta</h4>
                            <div class="review-stars">
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                            </div>
                        </div>
                    </div>
                    <p class="review-text">"Best smoothies in town! I order every morning before work. The Green Detox keeps me energized throughout the day. Great service too!"</p>
                    <small style="color: #999; display: block; margin-top: 15px;">
                        <i class="fas fa-calendar"></i> Jan 12, 2026
                    </small>
                </div>
                
                <div class="review-card">
                    <div class="review-header">
                        <img src="https://ui-avatars.com/api/?name=Ananya+K&background=27ae60&color=fff" alt="Ananya K" class="review-avatar">
                        <div class="review-info">
                            <h4>Ananya Kumar</h4>
                            <div class="review-stars">
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star-half-o"></i>
                            </div>
                        </div>
                    </div>
                    <p class="review-text">"Finally a place that serves healthy food without compromising on taste. Their overnight oats are amazing! Perfect for breakfast."</p>
                    <small style="color: #999; display: block; margin-top: 15px;">
                        <i class="fas fa-calendar"></i> Jan 10, 2026
                    </small>
                </div>
                
                <div class="review-card">
                    <div class="review-header">
                        <img src="https://ui-avatars.com/api/?name=Vikram+P&background=3498db&color=fff" alt="Vikram P" class="review-avatar">
                        <div class="review-info">
                            <h4>Vikram Patel</h4>
                            <div class="review-stars">
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                            </div>
                        </div>
                    </div>
                    <p class="review-text">"The quality of fruits is exceptional. You can really taste the freshness. My family loves ordering from FRUITOLA. Highly recommended!"</p>
                    <small style="color: #999; display: block; margin-top: 15px;">
                        <i class="fas fa-calendar"></i> Jan 08, 2026
                    </small>
                </div>
                
                <div class="review-card">
                    <div class="review-header">
                        <img src="https://ui-avatars.com/api/?name=Sneha+R&background=9b59b6&color=fff" alt="Sneha R" class="review-avatar">
                        <div class="review-info">
                            <h4>Sneha Reddy</h4>
                            <div class="review-stars">
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                            </div>
                        </div>
                    </div>
                    <p class="review-text">"I'm a regular customer now. The Berry Blast smoothie is my go-to drink. Quick delivery and excellent packaging. Keep up the great work!"</p>
                    <small style="color: #999; display: block; margin-top: 15px;">
                        <i class="fas fa-calendar"></i> Jan 05, 2026
                    </small>
                </div>
                
                <div class="review-card">
                    <div class="review-header">
                        <img src="https://ui-avatars.com/api/?name=Arjun+G&background=e67e22&color=fff" alt="Arjun G" class="review-avatar">
                        <div class="review-info">
                            <h4>Arjun Gupta</h4>
                            <div class="review-stars">
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star-o"></i>
                            </div>
                        </div>
                    </div>
                    <p class="review-text">"Great place for healthy snacking options. The sandwiches are fresh and tasty. Would love to see more variety in the menu!"</p>
                    <small style="color: #999; display: block; margin-top: 15px;">
                        <i class="fas fa-calendar"></i> Jan 03, 2026
                    </small>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="cta-content">
            <h2 class="cta-title">Ready to Try Our Delicious Menu?</h2>
            <p class="cta-text">Order now and experience the freshness yourself!</p>
            <div class="cta-buttons">
                <a href="menu.php" class="btn btn-white">
                    <i class="fas fa-utensils"></i> View Menu
                </a>
                <?php if (!empty($settings['whatsapp_number'])): ?>
                <a href="https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $settings['whatsapp_number']); ?>" class="btn btn-outline-white" target="_blank">
                    <i class="fab fa-whatsapp"></i> Order Now
                </a>
                <?php endif; ?>
            </div>
        </div>
    </section>

    <!-- Footer -->
    <footer class="footer">
        <div class="footer-container">
            <div class="footer-brand">
                <h2 class="logo-text">FRUITOLA</h2>
                <p>Your daily fresh dose of nature's goodness. We serve premium fruit bowls, smoothies, and healthy snacks made with 100% real fruits.</p>
                <div class="footer-social">
                    <a href="#" class="social-icon"><i class="fab fa-facebook-f"></i></a>
                    <a href="#" class="social-icon"><i class="fab fa-instagram"></i></a>
                    <a href="#" class="social-icon"><i class="fab fa-twitter"></i></a>
                    <a href="#" class="social-icon"><i class="fab fa-youtube"></i></a>
                </div>
            </div>
            
            <div class="footer-links-section">
                <h3 class="footer-title">Quick Links</h3>
                <ul class="footer-links">
                    <li><a href="index.php">Home</a></li>
                    <li><a href="menu.php">Our Menu</a></li>
                    <li><a href="reviews.php">Reviews</a></li>
                    <li><a href="contact.php">Contact Us</a></li>
                    <li><a href="roadmap.php">Roadmap</a></li>
                </ul>
            </div>
            
            <div class="footer-links-section">
                <h3 class="footer-title">Categories</h3>
                <ul class="footer-links">
                    <li><a href="menu.php?category=fruit-salads">Fruit Salads</a></li>
                    <li><a href="menu.php?category=oats-bowls">Oats Bowls</a></li>
                    <li><a href="menu.php?category=smoothies">Smoothies</a></li>
                    <li><a href="menu.php?category=tea-coffee">Tea & Coffee</a></li>
                </ul>
            </div>
            
            <div class="footer-contact">
                <h3 class="footer-title">Contact Us</h3>
                <?php if (!empty($settings['phone_number'])): ?>
                <p><i class="fas fa-phone"></i> <?php echo htmlspecialchars($settings['phone_number']); ?></p>
                <?php endif; ?>
                <?php if (!empty($settings['whatsapp_number'])): ?>
                <p><i class="fab fa-whatsapp"></i> <?php echo htmlspecialchars($settings['whatsapp_number']); ?></p>
                <?php endif; ?>
                <?php if (!empty($settings['admin_email'])): ?>
                <p><i class="fas fa-envelope"></i> <?php echo htmlspecialchars($settings['admin_email']); ?></p>
                <?php endif; ?>
                <?php if (!empty($settings['address'])): ?>
                <p><i class="fas fa-map-marker-alt"></i> <?php echo htmlspecialchars($settings['address']); ?></p>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="footer-bottom">
            <p>&copy; <?php echo date('Y'); ?> FRUITOLA - Daily Fresh Dose. All rights reserved.</p>
        </div>
    </footer>

    <script>
        function toggleMenu() {
            document.getElementById('navLinks').classList.toggle('active');
        }
        
        window.addEventListener('scroll', function() {
            const navbar = document.querySelector('.navbar');
            if (window.scrollY > 50) {
                navbar.style.boxShadow = '0 5px 30px rgba(0,0,0,0.1)';
            } else {
                navbar.style.boxShadow = '0 2px 30px rgba(0,0,0,0.1)';
            }
        });
        
        // Star rating functionality
        const stars = document.querySelectorAll('.star-rating-input .star');
        const ratingInput = document.getElementById('ratingValue');
        
        stars.forEach(star => {
            star.addEventListener('click', function() {
                const rating = this.getAttribute('data-rating');
                ratingInput.value = rating;
                
                stars.forEach(s => {
                    if (s.getAttribute('data-rating') <= rating) {
                        s.classList.add('active');
                    } else {
                        s.classList.remove('active');
                    }
                });
            });
            
            star.addEventListener('mouseenter', function() {
                const rating = this.getAttribute('data-rating');
                stars.forEach(s => {
                    if (s.getAttribute('data-rating') <= rating) {
                        s.style.color = '#f39c12';
                    } else {
                        s.style.color = '#ddd';
                    }
                });
            });
        });
        
        document.querySelector('.star-rating-input').addEventListener('mouseleave', function() {
            const currentRating = ratingInput.value;
            stars.forEach(s => {
                if (s.getAttribute('data-rating') <= currentRating) {
                    s.style.color = '#f39c12';
                } else {
                    s.style.color = '#ddd';
                }
            });
        });
    </script>
</body>
</html>
