<?php
require_once '../config/db.php';

if (!isAdminLoggedIn()) {
    redirect('login.php');
}

$conn = getDBConnection();

// Handle delete
if (isset($_GET['delete'])) {
    $id = intval($_GET['delete']);
    $stmt = $conn->prepare("DELETE FROM menu_items WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        setFlashMessage('success', 'Menu item deleted successfully!');
    } else {
        setFlashMessage('error', 'Error deleting menu item.');
    }
    redirect('menu-manage.php');
}

// Handle toggle active
if (isset($_GET['toggle'])) {
    $id = intval($_GET['toggle']);
    $stmt = $conn->prepare("UPDATE menu_items SET active = NOT active WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    redirect('menu-manage.php');
}

// Get all menu items with categories
$menu_items = [];
$result = $conn->query("SELECT m.*, c.name as category_name FROM menu_items m JOIN categories c ON m.category_id = c.id ORDER BY m.created_at DESC");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $menu_items[] = $row;
    }
}

$conn->close();

$flash = getFlashMessage();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Menu - FRUITOLA Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .admin-body {
            background: #f5f7fa;
            min-height: 100vh;
        }
        
        .admin-wrapper {
            display: flex;
        }
        
        .admin-sidebar {
            width: 280px;
            background: #2c3e50;
            min-height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            padding: 30px 20px;
            overflow-y: auto;
        }
        
        .admin-logo {
            text-align: center;
            margin-bottom: 40px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .admin-logo h2 {
            color: white;
            font-size: 1.8rem;
        }
        
        .admin-logo span {
            color: #f39c12;
        }
        
        .admin-nav {
            list-style: none;
        }
        
        .admin-nav li {
            margin-bottom: 5px;
        }
        
        .admin-nav a {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 15px 20px;
            color: rgba(255,255,255,0.7);
            text-decoration: none;
            border-radius: 12px;
            transition: all 0.3s ease;
        }
        
        .admin-nav a:hover,
        .admin-nav a.active {
            background: #e74c3c;
            color: white;
        }
        
        .admin-nav a i {
            width: 20px;
        }
        
        .admin-main {
            margin-left: 280px;
            flex: 1;
            padding: 30px;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .admin-title {
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .admin-user {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .admin-user a {
            color: #e74c3c;
            text-decoration: none;
        }
        
        .btn-add {
            background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
            color: white;
            padding: 12px 25px;
            border-radius: 10px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-add:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(39, 174, 96, 0.3);
        }
        
        .table-container {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.05);
            overflow-x: auto;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table th,
        .data-table td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        .data-table th {
            font-weight: 700;
            color: #666;
            background: #f8f9fa;
        }
        
        .data-table tr:hover {
            background: #f8f9fa;
        }
        
        .item-image {
            width: 60px;
            height: 60px;
            border-radius: 10px;
            object-fit: cover;
        }
        
        .btn-edit {
            background: #3498db;
            color: white;
            padding: 8px 15px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 0.85rem;
            margin-right: 5px;
        }
        
        .btn-delete {
            background: #e74c3c;
            color: white;
            padding: 8px 15px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 0.85rem;
        }
        
        .btn-toggle {
            background: #95a5a6;
            color: white;
            padding: 8px 15px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 0.85rem;
            margin-right: 5px;
        }
        
        .btn-toggle.active {
            background: #27ae60;
        }
        
        .status-badge {
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        
        .status-active { background: #d4edda; color: #155724; }
        .status-inactive { background: #f8d7da; color: #721c24; }
        
        .featured-badge {
            background: #f39c12;
            color: white;
            padding: 3px 10px;
            border-radius: 10px;
            font-size: 0.7rem;
            margin-left: 10px;
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 12px;
            margin-bottom: 25px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        @media (max-width: 768px) {
            .admin-sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            
            .admin-main {
                margin-left: 0;
            }
            
            .admin-wrapper {
                flex-direction: column;
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <aside class="admin-sidebar">
            <div class="admin-logo">
                <h2>FRUITOLA <span>Admin</span></h2>
            </div>
            
            <ul class="admin-nav">
                <li><a href="dashboard.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
                <li><a href="menu-add.php"><i class="fas fa-plus-circle"></i> Add Menu Item</a></li>
                <li><a href="menu-manage.php" class="active"><i class="fas fa-utensils"></i> Manage Menu</a></li>
                <li><a href="reviews-manage.php"><i class="fas fa-star"></i> Reviews</a></li>
                <li><a href="messages.php"><i class="fas fa-envelope"></i> Messages</a></li>
                <li><a href="email-settings.php"><i class="fas fa-cog"></i> Settings</a></li>
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
            </ul>
        </aside>
        
        <!-- Main Content -->
        <main class="admin-main">
            <div class="admin-header">
                <h1 class="admin-title">Manage Menu</h1>
                <div class="admin-user">
                    <a href="menu-add.php" class="btn-add"><i class="fas fa-plus"></i> Add New Item</a>
                    <span style="margin-left: 20px;"><i class="fas fa-user"></i> <?php echo htmlspecialchars($_SESSION['admin_email'] ?? 'Admin'); ?></span>
                    <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                </div>
            </div>
            
            <?php if ($flash): ?>
                <div class="alert alert-<?php echo $flash['type']; ?>">
                    <i class="fas fa-<?php echo $flash['type'] == 'success' ? 'check-circle' : 'exclamation-circle'; ?>"></i> <?php echo $flash['message']; ?>
                </div>
            <?php endif; ?>
            
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Image</th>
                            <th>Name</th>
                            <th>Category</th>
                            <th>Price</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($menu_items)): ?>
                            <?php foreach ($menu_items as $item): ?>
                            <tr>
                                <td>
                                    <img src="<?php echo $item['image'] ? '../assets/uploads/' . $item['image'] : 'https://via.placeholder.com/60'; ?>" alt="<?php echo htmlspecialchars($item['name']); ?>" class="item-image">
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($item['name']); ?>
                                    <?php if ($item['featured']): ?>
                                    <span class="featured-badge"><i class="fas fa-star"></i> Featured</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($item['category_name']); ?></td>
                                <td>₹<?php echo number_format($item['price'], 2); ?></td>
                                <td><span class="status-badge status-<?php echo $item['active'] ? 'active' : 'inactive'; ?>"><?php echo $item['active'] ? 'Active' : 'Inactive'; ?></span></td>
                                <td>
                                    <a href="menu-edit.php?id=<?php echo $item['id']; ?>" class="btn-edit"><i class="fas fa-edit"></i> Edit</a>
                                    <a href="?toggle=<?php echo $item['id']; ?>" class="btn-toggle <?php echo $item['active'] ? 'active' : ''; ?>"><i class="fas fa-power-off"></i></a>
                                    <a href="?delete=<?php echo $item['id']; ?>" class="btn-delete" onclick="return confirm('Are you sure you want to delete this item?')"><i class="fas fa-trash"></i></a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="6" style="text-align: center; color: #666; padding: 40px;">
                                    <i class="fas fa-utensils" style="font-size: 3rem; margin-bottom: 15px; display: block;"></i>
                                    No menu items found. <a href="menu-add.php" style="color: #e74c3c;">Add your first item</a>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
</body>
</html>
