<?php
require_once '../config/db.php';

if (!isAdminLoggedIn()) {
    redirect('login.php');
}

$conn = getDBConnection();
$success = '';
$error = '';

// Get item ID
$id = intval($_GET['id'] ?? 0);
if (!$id) {
    redirect('menu-manage.php');
}

// Get categories
$categories = [];
$result = $conn->query("SELECT * FROM categories ORDER BY display_order");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
}

// Get item details
$stmt = $conn->prepare("SELECT * FROM menu_items WHERE id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();
$result = $stmt->get_result();
$item = $result->fetch_assoc();

if (!$item) {
    redirect('menu-manage.php');
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $category_id = intval($_POST['category_id'] ?? 0);
    $name = sanitize($conn, $_POST['name'] ?? '');
    $description = sanitize($conn, $_POST['description'] ?? '');
    $price = floatval($_POST['price'] ?? 0);
    $featured = isset($_POST['featured']) ? 1 : 0;
    
    // Handle image upload
    $image_path = $item['image'];
    if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['image']['name'];
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        if (in_array($ext, $allowed)) {
            $new_filename = uniqid() . '.' . $ext;
            $upload_path = '../assets/uploads/menu/';
            
            if (!is_dir($upload_path)) {
                mkdir($upload_path, 0777, true);
            }
            
            if (move_uploaded_file($_FILES['image']['tmp_name'], $upload_path . $new_filename)) {
                // Delete old image
                if ($image_path && file_exists('../assets/uploads/' . $image_path)) {
                    unlink('../assets/uploads/' . $image_path);
                }
                $image_path = 'menu/' . $new_filename;
            }
        }
    }
    
    if (empty($name) || $category_id == 0 || $price <= 0) {
        $error = "Please fill in all required fields.";
    } else {
        $stmt = $conn->prepare("UPDATE menu_items SET category_id = ?, name = ?, description = ?, price = ?, image = ?, featured = ? WHERE id = ?");
        $stmt->bind_param("issdsii", $category_id, $name, $description, $price, $image_path, $featured, $id);
        
        if ($stmt->execute()) {
            setFlashMessage('success', 'Menu item updated successfully!');
            redirect('menu-manage.php');
        } else {
            $error = "Error updating menu item: " . $conn->error;
        }
    }
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Menu Item - FRUITOLA Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .admin-body {
            background: #f5f7fa;
            min-height: 100vh;
        }
        
        .admin-wrapper {
            display: flex;
        }
        
        .admin-sidebar {
            width: 280px;
            background: #2c3e50;
            min-height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            padding: 30px 20px;
            overflow-y: auto;
        }
        
        .admin-logo {
            text-align: center;
            margin-bottom: 40px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .admin-logo h2 {
            color: white;
            font-size: 1.8rem;
        }
        
        .admin-logo span {
            color: #f39c12;
        }
        
        .admin-nav {
            list-style: none;
        }
        
        .admin-nav li {
            margin-bottom: 5px;
        }
        
        .admin-nav a {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 15px 20px;
            color: rgba(255,255,255,0.7);
            text-decoration: none;
            border-radius: 12px;
            transition: all 0.3s ease;
        }
        
        .admin-nav a:hover,
        .admin-nav a.active {
            background: #e74c3c;
            color: white;
        }
        
        .admin-nav a i {
            width: 20px;
        }
        
        .admin-main {
            margin-left: 280px;
            flex: 1;
            padding: 30px;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .admin-title {
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .admin-user {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .admin-user a {
            color: #e74c3c;
            text-decoration: none;
        }
        
        .form-container {
            background: white;
            border-radius: 20px;
            padding: 40px;
            max-width: 800px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.05);
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 10px;
            font-weight: 600;
            color: #333;
        }
        
        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 15px 20px;
            border: 2px solid #e0e0e0;
            border-radius: 12px;
            font-size: 1rem;
            font-family: inherit;
            transition: all 0.3s ease;
        }
        
        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: #e74c3c;
            box-shadow: 0 0 0 4px rgba(231, 76, 60, 0.1);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 25px;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .checkbox-group input {
            width: auto;
        }
        
        .current-image {
            max-width: 200px;
            border-radius: 10px;
            margin-top: 10px;
        }
        
        .btn-submit {
            background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
            color: white;
            border: none;
            padding: 15px 40px;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-submit:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(52, 152, 219, 0.3);
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 12px;
            margin-bottom: 25px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        @media (max-width: 768px) {
            .admin-sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            
            .admin-main {
                margin-left: 0;
            }
            
            .admin-wrapper {
                flex-direction: column;
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <aside class="admin-sidebar">
            <div class="admin-logo">
                <h2>FRUITOLA <span>Admin</span></h2>
            </div>
            
            <ul class="admin-nav">
                <li><a href="dashboard.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
                <li><a href="menu-add.php"><i class="fas fa-plus-circle"></i> Add Menu Item</a></li>
                <li><a href="menu-manage.php" class="active"><i class="fas fa-utensils"></i> Manage Menu</a></li>
                <li><a href="reviews-manage.php"><i class="fas fa-star"></i> Reviews</a></li>
                <li><a href="messages.php"><i class="fas fa-envelope"></i> Messages</a></li>
                <li><a href="email-settings.php"><i class="fas fa-cog"></i> Settings</a></li>
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
            </ul>
        </aside>
        
        <!-- Main Content -->
        <main class="admin-main">
            <div class="admin-header">
                <h1 class="admin-title">Edit Menu Item</h1>
                <div class="admin-user">
                    <a href="menu-manage.php" style="margin-right: 20px;"><i class="fas fa-arrow-left"></i> Back to Menu</a>
                    <span><i class="fas fa-user"></i> <?php echo htmlspecialchars($_SESSION['admin_email'] ?? 'Admin'); ?></span>
                    <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                </div>
            </div>
            
            <div class="form-container">
                <?php if ($error): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
                    </div>
                <?php endif; ?>
                
                <form method="POST" action="" enctype="multipart/form-data">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Category *</label>
                            <select name="category_id" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo $cat['id']; ?>" <?php echo $item['category_id'] == $cat['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($cat['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Item Name *</label>
                            <input type="text" name="name" required placeholder="Enter item name" value="<?php echo htmlspecialchars($item['name']); ?>">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Description</label>
                        <textarea name="description" rows="4" placeholder="Enter item description"><?php echo htmlspecialchars($item['description']); ?></textarea>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Price (₹) *</label>
                            <input type="number" name="price" step="0.01" min="0" required placeholder="Enter price" value="<?php echo $item['price']; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label>Image</label>
                            <input type="file" name="image" accept="image/*">
                            <?php if ($item['image']): ?>
                            <img src="../assets/uploads/<?php echo $item['image']; ?>" alt="Current Image" class="current-image">
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <div class="checkbox-group">
                            <input type="checkbox" name="featured" id="featured" value="1" <?php echo $item['featured'] ? 'checked' : ''; ?>>
                            <label for="featured" style="margin-bottom: 0;">Feature this item on homepage</label>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn-submit">
                        <i class="fas fa-save"></i> Update Menu Item
                    </button>
                </form>
            </div>
        </main>
    </div>
</body>
</html>
