<?php
require_once '../config/db.php';

if (!isAdminLoggedIn()) {
    redirect('login.php');
}

$conn = getDBConnection();

// Handle approve/reject
if (isset($_GET['action']) && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    $action = $_GET['action'];
    
    if ($action == 'approve') {
        $stmt = $conn->prepare("UPDATE reviews SET status = 'approved' WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        setFlashMessage('success', 'Review approved successfully!');
    } elseif ($action == 'reject') {
        $stmt = $conn->prepare("UPDATE reviews SET status = 'rejected' WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        setFlashMessage('success', 'Review rejected successfully!');
    } elseif ($action == 'delete') {
        // Get image path first
        $stmt = $conn->prepare("SELECT image FROM reviews WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        $review = $result->fetch_assoc();
        
        // Delete image if exists
        if ($review && $review['image'] && file_exists('../assets/uploads/' . $review['image'])) {
            unlink('../assets/uploads/' . $review['image']);
        }
        
        $stmt = $conn->prepare("DELETE FROM reviews WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        setFlashMessage('success', 'Review deleted successfully!');
    }
    
    redirect('reviews-manage.php');
}

// Get filter
$status_filter = $_GET['status'] ?? '';

// Get reviews
$reviews = [];
if ($status_filter) {
    $stmt = $conn->prepare("SELECT * FROM reviews WHERE status = ? ORDER BY created_at DESC");
    $stmt->bind_param("s", $status_filter);
    $stmt->execute();
    $result = $stmt->get_result();
} else {
    $result = $conn->query("SELECT * FROM reviews ORDER BY created_at DESC");
}

if ($result) {
    while ($row = $result->fetch_assoc()) {
        $reviews[] = $row;
    }
}

$conn->close();

$flash = getFlashMessage();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Reviews - FRUITOLA Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .admin-body {
            background: #f5f7fa;
            min-height: 100vh;
        }
        
        .admin-wrapper {
            display: flex;
        }
        
        .admin-sidebar {
            width: 280px;
            background: #2c3e50;
            min-height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            padding: 30px 20px;
            overflow-y: auto;
        }
        
        .admin-logo {
            text-align: center;
            margin-bottom: 40px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .admin-logo h2 {
            color: white;
            font-size: 1.8rem;
        }
        
        .admin-logo span {
            color: #f39c12;
        }
        
        .admin-nav {
            list-style: none;
        }
        
        .admin-nav li {
            margin-bottom: 5px;
        }
        
        .admin-nav a {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 15px 20px;
            color: rgba(255,255,255,0.7);
            text-decoration: none;
            border-radius: 12px;
            transition: all 0.3s ease;
        }
        
        .admin-nav a:hover,
        .admin-nav a.active {
            background: #e74c3c;
            color: white;
        }
        
        .admin-nav a i {
            width: 20px;
        }
        
        .admin-main {
            margin-left: 280px;
            flex: 1;
            padding: 30px;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .admin-title {
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .admin-user {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .admin-user a {
            color: #e74c3c;
            text-decoration: none;
        }
        
        .filter-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 25px;
        }
        
        .filter-tab {
            padding: 10px 20px;
            border-radius: 10px;
            background: white;
            color: #333;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s ease;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .filter-tab:hover,
        .filter-tab.active {
            background: linear-gradient(135deg, #e74c3c 0%, #f39c12 100%);
            color: white;
        }
        
        .reviews-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 25px;
        }
        
        .review-card {
            background: white;
            border-radius: 20px;
            padding: 25px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.05);
            transition: all 0.3s ease;
        }
        
        .review-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        
        .review-header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .review-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
        }
        
        .review-info h4 {
            font-size: 1.1rem;
            margin-bottom: 5px;
        }
        
        .review-stars {
            color: #f39c12;
        }
        
        .review-text {
            color: #666;
            line-height: 1.6;
            margin-bottom: 15px;
        }
        
        .review-image {
            max-width: 100%;
            border-radius: 10px;
            margin-bottom: 15px;
        }
        
        .review-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-top: 15px;
            border-top: 1px solid #eee;
        }
        
        .review-date {
            color: #999;
            font-size: 0.85rem;
        }
        
        .review-actions {
            display: flex;
            gap: 10px;
        }
        
        .btn-approve {
            background: #27ae60;
            color: white;
            padding: 8px 15px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 0.85rem;
        }
        
        .btn-reject {
            background: #f39c12;
            color: white;
            padding: 8px 15px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 0.85rem;
        }
        
        .btn-delete {
            background: #e74c3c;
            color: white;
            padding: 8px 15px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 0.85rem;
        }
        
        .status-badge {
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        
        .status-pending { background: #fff3cd; color: #856404; }
        .status-approved { background: #d4edda; color: #155724; }
        .status-rejected { background: #f8d7da; color: #721c24; }
        
        .alert {
            padding: 15px 20px;
            border-radius: 12px;
            margin-bottom: 25px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        @media (max-width: 768px) {
            .admin-sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            
            .admin-main {
                margin-left: 0;
            }
            
            .admin-wrapper {
                flex-direction: column;
            }
            
            .reviews-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <aside class="admin-sidebar">
            <div class="admin-logo">
                <h2>FRUITOLA <span>Admin</span></h2>
            </div>
            
            <ul class="admin-nav">
                <li><a href="dashboard.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
                <li><a href="menu-add.php"><i class="fas fa-plus-circle"></i> Add Menu Item</a></li>
                <li><a href="menu-manage.php"><i class="fas fa-utensils"></i> Manage Menu</a></li>
                <li><a href="reviews-manage.php" class="active"><i class="fas fa-star"></i> Reviews</a></li>
                <li><a href="messages.php"><i class="fas fa-envelope"></i> Messages</a></li>
                <li><a href="email-settings.php"><i class="fas fa-cog"></i> Settings</a></li>
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
            </ul>
        </aside>
        
        <!-- Main Content -->
        <main class="admin-main">
            <div class="admin-header">
                <h1 class="admin-title">Manage Reviews</h1>
                <div class="admin-user">
                    <span><i class="fas fa-user"></i> <?php echo htmlspecialchars($_SESSION['admin_email'] ?? 'Admin'); ?></span>
                    <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                </div>
            </div>
            
            <?php if ($flash): ?>
                <div class="alert alert-<?php echo $flash['type']; ?>">
                    <i class="fas fa-<?php echo $flash['type'] == 'success' ? 'check-circle' : 'exclamation-circle'; ?>"></i> <?php echo $flash['message']; ?>
                </div>
            <?php endif; ?>
            
            <!-- Filter Tabs -->
            <div class="filter-tabs">
                <a href="reviews-manage.php" class="filter-tab <?php echo empty($status_filter) ? 'active' : ''; ?>">All Reviews</a>
                <a href="?status=pending" class="filter-tab <?php echo $status_filter == 'pending' ? 'active' : ''; ?>">Pending</a>
                <a href="?status=approved" class="filter-tab <?php echo $status_filter == 'approved' ? 'active' : ''; ?>">Approved</a>
                <a href="?status=rejected" class="filter-tab <?php echo $status_filter == 'rejected' ? 'active' : ''; ?>">Rejected</a>
            </div>
            
            <!-- Reviews Grid -->
            <div class="reviews-grid">
                <?php if (!empty($reviews)): ?>
                    <?php foreach ($reviews as $review): ?>
                    <div class="review-card">
                        <div class="review-header">
                            <img src="https://ui-avatars.com/api/?name=<?php echo urlencode($review['customer_name']); ?>&background=f39c12&color=fff" alt="<?php echo htmlspecialchars($review['customer_name']); ?>" class="review-avatar">
                            <div class="review-info">
                                <h4><?php echo htmlspecialchars($review['customer_name']); ?></h4>
                                <div class="review-stars">
                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                        <i class="fas fa-star<?php echo $i <= $review['rating'] ? '' : '-o'; ?>"></i>
                                    <?php endfor; ?>
                                </div>
                            </div>
                            <span class="status-badge status-<?php echo $review['status']; ?>" style="margin-left: auto;"><?php echo ucfirst($review['status']); ?></span>
                        </div>
                        
                        <p class="review-text">"<?php echo htmlspecialchars($review['review_text']); ?>"</p>
                        
                        <?php if ($review['image']): ?>
                        <img src="../assets/uploads/<?php echo $review['image']; ?>" alt="Review Image" class="review-image">
                        <?php endif; ?>
                        
                        <div class="review-footer">
                            <span class="review-date"><i class="fas fa-calendar"></i> <?php echo date('M d, Y', strtotime($review['created_at'])); ?></span>
                            <div class="review-actions">
                                <?php if ($review['status'] != 'approved'): ?>
                                <a href="?action=approve&id=<?php echo $review['id']; ?>" class="btn-approve"><i class="fas fa-check"></i> Approve</a>
                                <?php endif; ?>
                                <?php if ($review['status'] != 'rejected'): ?>
                                <a href="?action=reject&id=<?php echo $review['id']; ?>" class="btn-reject"><i class="fas fa-times"></i> Reject</a>
                                <?php endif; ?>
                                <a href="?action=delete&id=<?php echo $review['id']; ?>" class="btn-delete" onclick="return confirm('Are you sure you want to delete this review?')"><i class="fas fa-trash"></i></a>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div style="grid-column: 1/-1; text-align: center; padding: 60px; background: white; border-radius: 20px;">
                        <i class="fas fa-star" style="font-size: 4rem; color: #ddd; margin-bottom: 20px;"></i>
                        <h3>No reviews found</h3>
                        <p style="color: #666;">Reviews will appear here once customers submit them.</p>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
</body>
</html>
