<?php
require_once '../config/db.php';

if (!isAdminLoggedIn()) {
    redirect('login.php');
}

$conn = getDBConnection();
$success = '';
$error = '';

// Get current settings
$settings = [];
$result = $conn->query("SELECT * FROM site_settings");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $updates = [
        'site_name' => $_POST['site_name'] ?? '',
        'site_tagline' => $_POST['site_tagline'] ?? '',
        'admin_email' => $_POST['admin_email'] ?? '',
        'whatsapp_number' => $_POST['whatsapp_number'] ?? '',
        'phone_number' => $_POST['phone_number'] ?? '',
        'address' => $_POST['address'] ?? '',
        'google_map' => $_POST['google_map'] ?? '',
        'smtp_host' => $_POST['smtp_host'] ?? '',
        'smtp_port' => $_POST['smtp_port'] ?? '',
        'smtp_user' => $_POST['smtp_user'] ?? '',
        'smtp_pass' => $_POST['smtp_pass'] ?? ''
    ];
    
    $stmt = $conn->prepare("UPDATE site_settings SET setting_value = ? WHERE setting_key = ?");
    
    foreach ($updates as $key => $value) {
        $stmt->bind_param("ss", $value, $key);
        $stmt->execute();
    }
    
    // Refresh settings
    $result = $conn->query("SELECT * FROM site_settings");
    if ($result) {
        $settings = [];
        while ($row = $result->fetch_assoc()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
    }
    
    $success = "Settings updated successfully!";
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - FRUITOLA Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .admin-body {
            background: #f5f7fa;
            min-height: 100vh;
        }
        
        .admin-wrapper {
            display: flex;
        }
        
        .admin-sidebar {
            width: 280px;
            background: #2c3e50;
            min-height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            padding: 30px 20px;
            overflow-y: auto;
        }
        
        .admin-logo {
            text-align: center;
            margin-bottom: 40px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .admin-logo h2 {
            color: white;
            font-size: 1.8rem;
        }
        
        .admin-logo span {
            color: #f39c12;
        }
        
        .admin-nav {
            list-style: none;
        }
        
        .admin-nav li {
            margin-bottom: 5px;
        }
        
        .admin-nav a {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 15px 20px;
            color: rgba(255,255,255,0.7);
            text-decoration: none;
            border-radius: 12px;
            transition: all 0.3s ease;
        }
        
        .admin-nav a:hover,
        .admin-nav a.active {
            background: #e74c3c;
            color: white;
        }
        
        .admin-nav a i {
            width: 20px;
        }
        
        .admin-main {
            margin-left: 280px;
            flex: 1;
            padding: 30px;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .admin-title {
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .admin-user {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .admin-user a {
            color: #e74c3c;
            text-decoration: none;
        }
        
        .form-container {
            background: white;
            border-radius: 20px;
            padding: 40px;
            max-width: 900px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.05);
        }
        
        .form-section {
            margin-bottom: 40px;
            padding-bottom: 30px;
            border-bottom: 1px solid #eee;
        }
        
        .form-section:last-child {
            border-bottom: none;
            margin-bottom: 0;
            padding-bottom: 0;
        }
        
        .form-section h3 {
            font-size: 1.3rem;
            margin-bottom: 25px;
            color: #333;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 10px;
            font-weight: 600;
            color: #333;
        }
        
        .form-group input,
        .form-group textarea {
            width: 100%;
            padding: 15px 20px;
            border: 2px solid #e0e0e0;
            border-radius: 12px;
            font-size: 1rem;
            font-family: inherit;
            transition: all 0.3s ease;
        }
        
        .form-group input:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #e74c3c;
            box-shadow: 0 0 0 4px rgba(231, 76, 60, 0.1);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 25px;
        }
        
        .btn-submit {
            background: linear-gradient(135deg, #e74c3c 0%, #f39c12 100%);
            color: white;
            border: none;
            padding: 15px 40px;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-submit:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(231, 76, 60, 0.3);
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 12px;
            margin-bottom: 25px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        @media (max-width: 768px) {
            .admin-sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            
            .admin-main {
                margin-left: 0;
            }
            
            .admin-wrapper {
                flex-direction: column;
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <aside class="admin-sidebar">
            <div class="admin-logo">
                <h2>FRUITOLA <span>Admin</span></h2>
            </div>
            
            <ul class="admin-nav">
                <li><a href="dashboard.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
                <li><a href="menu-add.php"><i class="fas fa-plus-circle"></i> Add Menu Item</a></li>
                <li><a href="menu-manage.php"><i class="fas fa-utensils"></i> Manage Menu</a></li>
                <li><a href="reviews-manage.php"><i class="fas fa-star"></i> Reviews</a></li>
                <li><a href="messages.php"><i class="fas fa-envelope"></i> Messages</a></li>
                <li><a href="email-settings.php" class="active"><i class="fas fa-cog"></i> Settings</a></li>
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
            </ul>
        </aside>
        
        <!-- Main Content -->
        <main class="admin-main">
            <div class="admin-header">
                <h1 class="admin-title">Site Settings</h1>
                <div class="admin-user">
                    <span><i class="fas fa-user"></i> <?php echo htmlspecialchars($_SESSION['admin_email'] ?? 'Admin'); ?></span>
                    <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                </div>
            </div>
            
            <div class="form-container">
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i> <?php echo $success; ?>
                    </div>
                <?php endif; ?>
                
                <form method="POST" action="">
                    <!-- General Settings -->
                    <div class="form-section">
                        <h3><i class="fas fa-info-circle" style="color: #3498db;"></i> General Settings</h3>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Site Name</label>
                                <input type="text" name="site_name" value="<?php echo htmlspecialchars($settings['site_name'] ?? 'FRUITOLA'); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label>Site Tagline</label>
                                <input type="text" name="site_tagline" value="<?php echo htmlspecialchars($settings['site_tagline'] ?? 'Daily Fresh Dose'); ?>">
                            </div>
                        </div>
                    </div>
                    
                    <!-- Contact Settings -->
                    <div class="form-section">
                        <h3><i class="fas fa-address-card" style="color: #27ae60;"></i> Contact Information</h3>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Admin Email</label>
                                <input type="email" name="admin_email" value="<?php echo htmlspecialchars($settings['admin_email'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label>Phone Number</label>
                                <input type="text" name="phone_number" value="<?php echo htmlspecialchars($settings['phone_number'] ?? ''); ?>">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>WhatsApp Number</label>
                                <input type="text" name="whatsapp_number" value="<?php echo htmlspecialchars($settings['whatsapp_number'] ?? ''); ?>">
                                <small style="color: #666;">Include country code (e.g., 919876543210)</small>
                            </div>
                            
                            <div class="form-group">
                                <label>Address</label>
                                <input type="text" name="address" value="<?php echo htmlspecialchars($settings['address'] ?? ''); ?>">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Google Map Embed Code</label>
                            <textarea name="google_map" rows="3" placeholder="Paste your Google Maps embed iframe code here"><?php echo htmlspecialchars($settings['google_map'] ?? ''); ?></textarea>
                        </div>
                    </div>
                    
                    <!-- SMTP Settings -->
                    <div class="form-section">
                        <h3><i class="fas fa-envelope" style="color: #e74c3c;"></i> Email (SMTP) Settings</h3>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>SMTP Host</label>
                                <input type="text" name="smtp_host" value="<?php echo htmlspecialchars($settings['smtp_host'] ?? ''); ?>">
                                <small style="color: #666;">e.g., mail.yourdomain.com</small>
                            </div>
                            
                            <div class="form-group">
                                <label>SMTP Port</label>
                                <input type="text" name="smtp_port" value="<?php echo htmlspecialchars($settings['smtp_port'] ?? '587'); ?>">
                                <small style="color: #666;">Usually 587 for TLS or 465 for SSL</small>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>SMTP Username</label>
                                <input type="text" name="smtp_user" value="<?php echo htmlspecialchars($settings['smtp_user'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label>SMTP Password</label>
                                <input type="password" name="smtp_pass" value="<?php echo htmlspecialchars($settings['smtp_pass'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn-submit">
                        <i class="fas fa-save"></i> Save Settings
                    </button>
                </form>
            </div>
        </main>
    </div>
</body>
</html>
